<?php
session_start();
require '../../../script/conexion.php';

ini_set('display_errors', 0);
error_reporting(E_ALL);

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    exit('Método no permitido');
}

function post($k, $default = '') {
    return trim($_POST[$k] ?? $default);
}

/* ---------------- Entradas ---------------- */
$nombre_usuario      = post('nombre_usuario');
$id_produccion       = post('id_produccion');
$numero_produccion   = post('numero_produccion');
$referencia_producto = post('referencia_producto');
$talla               = post('talla');

$cantidad            = post('cantidad');       // cantidad actual (según el form)
$cantidad_edit       = post('cantidad_edit');  // nueva cantidad

$accion_editar   = isset($_POST['editar']);
$accion_eliminar = isset($_POST['eliminar']);

$hoy = date("Y-m-d H:i:s");
$descripcion = "Id: {$id_produccion} numero: {$numero_produccion} Ref: {$referencia_producto} Talla: {$talla} Cant: {$cantidad} {$hoy}";

/* ---------------- Validaciones ---------------- */
$errores = [];

if ($nombre_usuario === '') $errores[] = 'Falta nombre_usuario';
if ($id_produccion === '' || !ctype_digit($id_produccion) || (int)$id_produccion <= 0) $errores[] = 'id_produccion inválido';
if ($referencia_producto === '') $errores[] = 'Falta referencia_producto';
if ($talla === '') $errores[] = 'Falta talla';

if ($cantidad === '' || !is_numeric($cantidad) || (int)$cantidad < 0) $errores[] = 'cantidad inválida';
$cantidad_int = (int)$cantidad;

if ($accion_editar) {
    if ($cantidad_edit === '' || !is_numeric($cantidad_edit) || (int)$cantidad_edit < 0) $errores[] = 'cantidad_edit inválida';
    $cantidad_edit_int = (int)$cantidad_edit;
} else {
    $cantidad_edit_int = 0;
}

if (!$accion_editar && !$accion_eliminar) $errores[] = 'No se pudo determinar la acción';

if ($errores) {
    http_response_code(400);
    exit('Error: ' . implode(', ', $errores));
}

$id_int = (int)$id_produccion;

try {
    $conn->begin_transaction();

    /* -------------------------------------------------------
       1) Stock disponible (bloqueo para evitar carreras)
    ------------------------------------------------------- */
    $stmtStock = $conn->prepare("
        SELECT COALESCE(cantidad,0) AS cantidad
        FROM referencia_disponible
        WHERE referencia_producto = ? AND talla = ?
        FOR UPDATE
    ");
    if (!$stmtStock) throw new Exception("Prepare stock: " . $conn->error);

    $stmtStock->bind_param("ss", $referencia_producto, $talla);
    $stmtStock->execute();
    $row2 = $stmtStock->get_result()->fetch_assoc();
    $stmtStock->close();

    $disponible = (int)($row2['cantidad'] ?? 0);

    /* -------------------------------------------------------
       2) Validación de integridad (tu misma regla)
    ------------------------------------------------------- */
    $balance_edit   = $cantidad_int - $cantidad_edit_int; // si baja, es positivo (devuelve stock)
    $balance_delete = $cantidad_int;

    if ($accion_editar) {
        $permitido = ($cantidad_edit_int >= $cantidad_int) || ($balance_edit <= $disponible);
        if (!$permitido) {
            $conn->rollback();
            header("Location: ../produccion-detalle.php?id_produccion={$id_int}&msg=" . urlencode("Acción denegada: compromete inventario"));
            exit;
        }
    }

    if ($accion_eliminar) {
        if ($balance_delete > $disponible) {
            $conn->rollback();
            header("Location: ../produccion-detalle.php?id_produccion={$id_int}&msg=" . urlencode("Acción denegada: compromete inventario"));
            exit;
        }
    }

    /* -------------------------------------------------------
       3) Log de eliminación (cuando eliminas o cambias)
          (tu lo haces tanto en editar como eliminar)
    ------------------------------------------------------- */
    $stmtLog = $conn->prepare("
        INSERT INTO registro_delete (nombre_usuario, origen, descripcion)
        VALUES (?,?,?)
    ");
    if (!$stmtLog) throw new Exception("Prepare log: " . $conn->error);

    $origen = 'Produccion detalle';
    $stmtLog->bind_param("sss", $nombre_usuario, $origen, $descripcion);
    if (!$stmtLog->execute()) throw new Exception("Insert log: " . $stmtLog->error);
    $stmtLog->close();

    /* -------------------------------------------------------
       4) Aplicar cambio en produccion_detalle
          - EDITAR: UPDATE (más seguro que borrar+insertar)
          - ELIMINAR: DELETE
    ------------------------------------------------------- */
    if ($accion_editar) {
        $stmtUp = $conn->prepare("
            UPDATE produccion_detalle
            SET cantidad = ?
            WHERE id_produccion = ? AND referencia_producto = ? AND talla = ?
            LIMIT 1
        ");
        if (!$stmtUp) throw new Exception("Prepare update detalle: " . $conn->error);

        $stmtUp->bind_param("iiss", $cantidad_edit_int, $id_int, $referencia_producto, $talla);
        if (!$stmtUp->execute()) throw new Exception("Execute update detalle: " . $stmtUp->error);
        $stmtUp->close();

    } else { // eliminar
        $stmtDel = $conn->prepare("
            DELETE FROM produccion_detalle
            WHERE id_produccion = ? AND referencia_producto = ? AND talla = ?
        ");
        if (!$stmtDel) throw new Exception("Prepare delete detalle: " . $conn->error);

        $stmtDel->bind_param("iss", $id_int, $referencia_producto, $talla);
        if (!$stmtDel->execute()) throw new Exception("Execute delete detalle: " . $stmtDel->error);
        $stmtDel->close();
    }

    /* -------------------------------------------------------
       5) Recalcular referencia (igual que tú)
    ------------------------------------------------------- */
    $class_referencia = $referencia_producto;
    $class_talla = $talla;
    require '../../../clases/class-referencia.php';

    $conn->commit();

    /* -------------------------------------------------------
       6) Redirect
    ------------------------------------------------------- */
    if ($accion_editar) {
        header("Location: ../produccion-detalle.php?id_produccion={$id_int}&referencia_producto=" . urlencode($referencia_producto));
    } else {
        header("Location: ../produccion-detalle.php?id_produccion={$id_int}");
    }
    exit;

} catch (Exception $e) {
    $conn->rollback();
    http_response_code(500);
    exit("Error: " . $e->getMessage());
}
