<?php
session_start();
require '../../../script/conexion.php';

ini_set('display_errors', 0);
error_reporting(E_ALL);

// Solo POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    exit('Método no permitido');
}

function post($k, $default = '') {
    return trim($_POST[$k] ?? $default);
}

$id_despacho         = post('id_despacho');
$estado              = post('estado');
$referencia_producto = post('referencia_producto');
$talla               = post('talla');
$cantidad            = post('cantidad');
$precio_unidad       = post('precio_unidad');

/* ---------------- Validaciones ---------------- */
$errores = [];

if ($id_despacho === '' || !ctype_digit($id_despacho) || (int)$id_despacho <= 0) $errores[] = 'id_despacho inválido';
if ($estado === '') $errores[] = 'Falta estado';
if ($referencia_producto === '') $errores[] = 'Falta referencia_producto';
if ($talla === '') $errores[] = 'Falta talla';

if ($cantidad === '' || !is_numeric($cantidad) || (int)$cantidad <= 0) $errores[] = 'Cantidad inválida';
$cantidad_int = (int)$cantidad;

if ($precio_unidad === '' || !is_numeric($precio_unidad) || (float)$precio_unidad < 0) $errores[] = 'Precio unidad inválido';
$precio_num = (float)$precio_unidad;

if ($errores) {
    http_response_code(400);
    exit('Error: ' . implode(', ', $errores));
}

$id_int = (int)$id_despacho;

try {
    $conn->begin_transaction();

    // 1) Validar que exista el despacho y traer nombre_tercero (solo lo que usas)
    $stmtD = $conn->prepare("SELECT nombre_tercero FROM despacho WHERE id = ? LIMIT 1");
    if (!$stmtD) throw new Exception("Prepare despacho: " . $conn->error);

    $stmtD->bind_param("i", $id_int);
    $stmtD->execute();
    $resD = $stmtD->get_result();
    $rowt = $resD->fetch_assoc();
    $stmtD->close();

    if (!$rowt) {
        throw new Exception("El despacho no existe");
    }

    // 2) Stock disponible (COALESCE) + lock para evitar carrera (InnoDB)
    $stmtS = $conn->prepare("
        SELECT COALESCE(SUM(cantidad),0) AS disponible
        FROM referencia_disponible
        WHERE referencia_producto = ? AND talla = ?
        FOR UPDATE
    ");
    if (!$stmtS) throw new Exception("Prepare stock: " . $conn->error);

    $stmtS->bind_param("ss", $referencia_producto, $talla);
    $stmtS->execute();
    $resS = $stmtS->get_result();
    $rowS = $resS->fetch_assoc();
    $stmtS->close();

    $disponible = (int)($rowS['disponible'] ?? 0);

    if ($disponible < $cantidad_int) {
        $conn->rollback();
        header("Location: ../alert-producto-disponibilidad.php?id_despacho=" . $id_int
            . "&referencia_producto=" . urlencode($referencia_producto)
            . "&talla=" . urlencode($talla)
            . "&disponibilidad=" . $disponible
        );
        exit;
    }

    // 3) Insert detalle (prepared)
    $stmtI = $conn->prepare("
        INSERT INTO despacho_detalle
        (id_despacho, referencia_producto, talla, cantidad, precio_unidad, estado)
        VALUES (?,?,?,?,?,?)
    ");
    if (!$stmtI) throw new Exception("Prepare insert detalle: " . $conn->error);

    // i s s i d s
    $stmtI->bind_param("issids", $id_int, $referencia_producto, $talla, $cantidad_int, $precio_num, $estado);

    if (!$stmtI->execute()) throw new Exception("Execute insert detalle: " . $stmtI->error);
    $stmtI->close();

    // 4) Actualizaciones de tus clases
    $class_referencia = $referencia_producto;
    $class_talla = $talla;
    require '../../../clases/class-referencia.php';

    $class_nombre_tercero_cierre = $rowt['nombre_tercero'];
    $class_referencia_cierre = $referencia_producto;
    $class_talla_cierre = $talla;
    require '../../../clases/class-cierre.php';

    $conn->commit();

    // PRG redirect (sin &&)
    header("Location: ../despacho-detalle.php?id_despacho=" . $id_int
        . "&referencia_producto=" . urlencode($referencia_producto)
        . "&precio=" . urlencode((string)$precio_num)
    );
    exit;

} catch (Exception $e) {
    $conn->rollback();
    http_response_code(500);
    exit("Error: " . $e->getMessage());
}
